<?php

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) exit;

class TRP_rockai_Translate_Translator extends TRP_Machine_Translator {
    /**
     * Send request to Google Translation API
     *
     * @param string $source_language       Translate from language
     * @param string $language_code         Translate to language
     * @param array $strings_array          Array of string to translate
     *
     * @return array|WP_Error               Response
     */
    public function send_request( $source_language, $language_code, $strings_array ){
        /* build our translation request */
        $referer = $this->get_referer();
        /* Due to url length restrictions we need so send a POST request faked as a GET request and send the strings in the body of the request and not in the URL */
        $post_data['texts'] = array_values($strings_array);
        $post_data['tl'] = $language_code;
        $post_data['sl'] = $source_language??'auto';
        $response = wp_remote_post( "https://ai-translate-p.rockapi.com/translate", array(
                'headers' => array(
                    //'X-HTTP-Method-Override' => 'GET', //this fakes a GET request
                    'timeout'                => 45,
                    'Referer'                => $referer,
                    'X-rockapi-Host'         => 'ai-translate-p.rockapi.com',
                    'x-rockapi-key'          => $this->get_api_key(),
                    'Content-Type'           => 'application/json'
                ),
                'body' => wp_json_encode($post_data),
            )
        );
        $trans_data['data'] = [];
        if ( is_array( $response ) && isset( $response['response'] ) ) {
            if(isset( $response['response']['code']) && $response['response']['code'] == 200){
                $translation_response = json_decode( $response['body'] ,true);
                if(is_array($translation_response['texts'])){
                    foreach ($translation_response['texts'] as $v){
                        $tmp_trans['translatedText'] = $v;
                        $trans_data['data']['translations'][] = $tmp_trans;
                    }
                }else{
                    $trans_data['data']['translations'][0]['translatedText'] = $translation_response['texts'];
                }
            }
            if(isset( $response['response']['code']) && $response['response']['code'] == 401){
                $response['response']['code'] = 200;
                $translation_response = json_decode( $response['body'] ,true);
                $trans_data['data']['translations'][0]['translatedText'] = $translation_response['message'];
                
            }
            
        }
        if(is_array($response) && isset($response['body'])){
            $response['body'] = json_encode($trans_data);
        }
        return $response;
    }
    

    /**
     * Returns an array with the API provided translations of the $new_strings array.
     *
     * @param array $new_strings                    array with the strings that need translation. The keys are the node number in the DOM so we need to preserve the m
     * @param string $target_language_code          language code of the language that we will be translating to. Not equal to the google language code
     * @param string $source_language_code          language code of the language that we will be translating from. Not equal to the google language code
     * @return array                                array with the translation strings and the preserved keys or an empty array if something went wrong
     */
    public function translate_array($new_strings, $target_language_code, $source_language_code = null ){
        
        if ( $source_language_code == null ){
            $source_language_code = $this->settings['default-language'];
        }
        
        if( empty( $new_strings ) || !$this->verify_request_parameters( $target_language_code, $source_language_code ) )
            return array();
        
        $source_language = $this->machine_translation_codes[$source_language_code];
        $target_language = $this->machine_translation_codes[$target_language_code];
        $translated_strings = array();

        /* split our strings that need translation in chunks of maximum 128 strings because Google Translate has a limit of 128 strings */
        $new_strings_chunks = array_chunk( $new_strings, 128, true );
        /* if there are more than 128 strings we make multiple requests */
        foreach( $new_strings_chunks as $new_strings_chunk ){
            $response = $this->send_request( $source_language, $target_language, $new_strings_chunk );
            // this is run only if "Log machine translation queries." is set to Yes.
            $this->machine_translator_logger->log(array(
                'strings'   => serialize( $new_strings_chunk),
                'response'  => serialize( $response ),
                'lang_source'  => $source_language,
                'lang_target'  => $target_language,
            ));

            /* analyze the response */
            if ( is_array( $response ) && ! is_wp_error( $response ) && isset( $response['response'] ) &&
                isset( $response['response']['code']) && $response['response']['code'] == 200 ) {

                $translation_response = json_decode( $response['body'] );
                if ( empty( $translation_response->error ) ) {
                    $this->machine_translator_logger->count_towards_quota( $new_strings_chunk );

                    /* if we have strings build the translation strings array and make sure we keep the original keys from $new_string */
                    $translations = ( empty( $translation_response->data->translations ) ) ? array() : $translation_response->data->translations;
                    $i            = 0;

                    foreach ( $new_strings_chunk as $key => $old_string ) {

                        if ( isset( $translations[ $i ] ) && !empty( $translations[ $i ]->translatedText ) ) {
                            $translated_strings[ $key ] = $translations[ $i ]->translatedText;
                        } else {
                            /*  In some cases when API doesn't have a translation for a particular string,
                            translation is returned empty instead of same string. Setting original string as translation
                            prevents TP from keep trying to submit same string for translation endlessly.  */
                            $translated_strings[ $key ] = $old_string;
                        }

                        $i++;

                    }
                }

                if( $this->machine_translator_logger->quota_exceeded() )
                    break;

            }
        }

        // will have the same indexes as $new_string or it will be an empty array if something went wrong
        return $translated_strings;
    }

    /**
     * Send a test request to verify if the functionality is working
     */
    public function test_request(){
        return true;
        return $this->send_request( 'en', 'zh', array( 'about' ) );

    }

    public function get_api_key(){
        return isset( $this->settings['trp_machine_translation_settings'], $this->settings['trp_machine_translation_settings']['rockai-translate-key'] ) ? $this->settings['trp_machine_translation_settings']['rockai-translate-key'] : false;

    }


    public function get_supported_languages(){
        $supported_languages = ['ab','ace','ach','af','ak','alz','am','ar','as','awa','ay','az','ba','ban','bbc','be','bem','bew','bg','bho','bik','bm','bn','br','bs','bts','btx','bua','ca','ceb','cgg','chm','ckb','cnh','co','crh','crs','cs','cv','cy','da','de','din','doi','dov','dv','dz','ee','el','en','eo','es','et','eu','fa','ff','fi','fj','fr','fy','ga','gaa','gd','gl','gn','gom','gu','ha','haw','he','hi','hil','hmn','hr','hrx','ht','hu','hy','id','ig','ilo','is','it','iw','ja','jv','jw','ka','kk','km','kn','ko','kri','ktu','ku','ky','la','lb','lg','li','lij','lmo','ln','lo','lt','ltg','luo','lus','lv','mai','mak','mg','mi','min','mk','ml','mn','mni-Mtei','mr','ms','ms-Arab','mt','my','ne','new','nl','no','nr','nso','nus','ny','oc','om','or','pa','pa-Arab','pag','pam','pap','pl','ps','pt','qu','rn','ro','rom','ru','rw','sa','scn','sd','sg','shn','si','sk','sl','sm','sn','so','sq','sr','ss','st','su','sv','sw','szl','ta','te','tet','tg','th','ti','tk','tl','tn','tr','ts','tt','ug','uk','ur','uz','vi','xh','yi','yo','yua','yue','zh','zh-CN','zh-TW','zu','fil'];
        return apply_filters( 'trp_add_rockai_supported_languages_to_the_array', $supported_languages );
    }

    public function add_rockai_supported_languages_that_are_not_returned_by_the_post_response($supported_language){
        $supported_language[] = 'fil';

        return $supported_language;
    }


    public function get_engine_specific_language_codes($languages){
        return $this->trp_languages->get_iso_codes($languages);
    }

    /*
     * Google does not support formality yet, but we need this for the machine translation tab to show the unsupported languages for formality
     */
    public function check_formality(){

        $formality_supported_languages = array();

        return $formality_supported_languages;
    }

    public function check_api_key_validity() {
        $machine_translator = $this;
        $translation_engine = $this->settings['trp_machine_translation_settings']['translation-engine'];
        $api_key            = $machine_translator->get_api_key();

        $is_error       = false;
        $return_message = '';

        if ( 'rockai_translate' === $translation_engine && $this->settings['trp_machine_translation_settings']['machine-translation'] === 'yes') {

            if ( isset( $this->correct_api_key ) && $this->correct_api_key != null ) {
                return $this->correct_api_key;
            }

            if ( empty( $api_key ) ) {
                $is_error       = true;
                $return_message = __( 'Please enter your rockai Translate key.', 'translatepress-multilingual' );
            } else {
                // Perform test.
                $response = $machine_translator->test_request();
                $code     = wp_remote_retrieve_response_code( $response );
                if ( 200 !== $code ) {
                    $is_error           = true;
                    $translate_response = trp_gt_response_codes( $code );
                    $return_message     = $translate_response['message'];
                }
            }
            $this->correct_api_key = array(
                'message' => $return_message,
                'error'   => $is_error,
            );
        }

        return array(
            'message' => $return_message,
            'error'   => $is_error,
        );
    }
}
